local Cosmetic = require("defs.cosmetics.cosmetics")
local krandom = require("util.krandom")
local lume = require("util.lume")

local CharacterCreator = Class(function(self, inst, use_playerdata_storage)
	self.inst = inst
	self.isnew = true
	self.bodyparts = {}
	self.colorgroups = {}
	self.filtertags = {}
	self.symboltags = {}

	-- for colorgroup in pairs(Cosmetic.ColorGroups) do
	-- 	local colors = self:GetColorList(colorgroup)
	-- 	self:SetColorGroup(colorgroup, colors[1].name)
	-- end

	self:SetSpecies("ogre") -- TODO: optimization, remove this to reduce unnecessary churn

	-- deliberate assignment after setting "defaults"
	self.use_playerdata_storage = use_playerdata_storage

	self._onplayerdatachanged = function(_inst, data_changed_flags)
		self:_OnPlayerDataChanged(data_changed_flags)
	end
	self.inst:ListenForEvent("playerdatachanged", self._onplayerdatachanged)

	-- TODO: optimization, explore using an earlier event
	self._onplayerentered = function(_inst)
		TheLog.ch.CharacterCreator:printf("PlayerEntered: %s", _inst)
		self:_RefreshPlayerData()
	end
	self.inst:ListenForEvent("playerentered", self._onplayerentered)
end)


function CharacterCreator:OnRemoveFromEntity()
	self.inst:RemoveEventCallback("playerdatachanged", self._onplayerdatachanged)
	self.inst:RemoveEventCallback("playerentered", self._onplayerentered)
end

function CharacterCreator:GetSpecies()
	return self.species
end

function CharacterCreator:SetSpecies(species)
	if self:GetSpecies() == species then
		return
	end

	self.species = species
	if self.use_playerdata_storage then
		ThePlayerData:SetCharacterCreatorSpecies(self.inst.Network:GetPlayerID(), species)
	end

	if species ~= "canine" then
		self:SetBodyPart("OTHER", nil)
	end

	for i, data in ipairs(DEFAULT_CHARACTERS_SETUP) do
		if data.species == species then
			self:LoadFromTable(data)
		end
	end
end

function CharacterCreator:IsNew()
	return self.isnew
end

-- The owner parameter is a Bit of a hack since we need to check if the player has unlocked something, but we might be randomizing a puppet
function CharacterCreator:IsBodyPartUnlocked(def, owner)
	owner = owner or self.inst
	if owner.components.unlocktracker then
		return owner.components.unlocktracker:IsCosmeticUnlocked(def.name, "PLAYER_BODYPART")
	else -- A nil unlocktracker means this is a new character so we check the filtertags instead
		return def.filtertags.default_unlocked ~= nil
	end
end

function CharacterCreator:IsColorUnlocked(def, owner)
	owner = owner or self.inst
	if owner.components.unlocktracker then
		return owner.components.unlocktracker:IsCosmeticUnlocked(def.name, "PLAYER_COLOR")
	else -- A nil unlocktracker means this is a new character so we check the filtertags instead
		return def.filtertags.default_unlocked ~= nil
	end
end

function CharacterCreator:GetBodyPartList(bodypart, filtered)
	return Cosmetic.GetBodyPartList(bodypart, filtered and self.filtertags or nil)
end

function CharacterCreator:GetBodyPart(bodypart)
	return self.bodyparts[bodypart]
end

function CharacterCreator:ClearAllExcept(exception)
	local bodyparts = Cosmetic.BodyPartGroups
	for bodypart in pairs(bodyparts) do
		if bodypart ~= exception then
			self:ClearBodyPartSymbols(bodypart)
		end
	end
end

function CharacterCreator:ClearAllExceptTable(exceptions)
	local bodyparts = Cosmetic.BodyPartGroups
	for bodypart in pairs(bodyparts) do
		if not table.contains(exceptions, bodypart) then
			self:ClearBodyPartSymbols(bodypart)
		end
	end
end

function CharacterCreator:ValidateColorGroups()
	-- filter color groups
	local isColorMatched = function(color, tags, matchAll)
		matchAll = matchAll or false
		if not color.filtertags then
			return true
		end
		if not matchAll then
			if not tags then
				return true
			else
				for tag, _v in pairs(tags) do
					if color.filtertags[tag] then
						return true
					end
				end
			end
			return false
		else
			if not tags then
				return false
			else
				for tag, _v in pairs(tags) do
					if not color.filtertags[tag] then
						return false
					end
				end
			end
			return true
		end
	end

	for groupId, colorname in pairs(self.colorgroups) do
		local currentBodyPartColor = Cosmetic.Colors[groupId][colorname]
		if currentBodyPartColor and not isColorMatched(currentBodyPartColor, self.filtertags, false) then
			TheLog.ch.CharacterCreatorSpam:printf("Color group=%s id=%s is invalid for filter tags %s", groupId, colorname, tabletoordereddictstring(self.filtertags))
			local bodyPartColors = Cosmetic.Colors[groupId]
			local newFilterTags = deepcopy(self.filtertags)
			newFilterTags["default"] = true
			local foundDefault
			for newname, newColor in pairs(bodyPartColors) do
				if isColorMatched(newColor, newFilterTags, true) then
					TheLog.ch.CharacterCreatorSpam:printf("Setting default color group=%s id=%s", groupId, newname)
					self:SetColorGroup(groupId, newname)
					foundDefault = true
					break
				end
			end
			if not foundDefault then
				TheLog.ch.CharacterCreatorSpam:printf("Removing invalid color group=%s id=%s", groupId, colorname)
				self:SetColorGroup(groupId, nil)
			end
		end
	end
end

function CharacterCreator:SetBodyPart(bodypart, name)
	if self.bodyparts[bodypart] == name then
		return false, false
	end

	local items = Cosmetic.BodyParts[bodypart]
	if items == nil then
		TheLog.ch.CharacterCreator:printf("Invalid body part: %s", bodypart)
		return
	end

	local def
	if name then
		def = items[name]
		if not def then
			TheLog.ch.CharacterCreator:printf("Invalid %s item: %s", bodypart, name)
			return
		end
	end

	local oldpart = self.bodyparts[bodypart]
	if def ~= nil then
		self.bodyparts[bodypart] = name
		self:OverrideBodyPartSymbols(bodypart, def.build, def.colorgroup)
	else
		self.bodyparts[bodypart] = nil
		self:ClearBodyPartSymbols(bodypart)
	end

	--Reset filter tags
	local oldfiltertags = self.filtertags
	self.filtertags = {}

	for tag in pairs(def ~= nil and def.filtertags or oldfiltertags) do
		if tag ~= "default_unlocked" then
			self.filtertags[tag] = true
		end
	end

	--Reset symbol tags
	local oldsymboltags = {}
	oldsymboltags[bodypart] = deepcopy(self.symboltags[bodypart])
	self.symboltags[bodypart] = {}

	if def ~= nil and def.symboltags ~= nil then
		for tag in pairs(def.symboltags) do
			self.symboltags[bodypart][tag] = true
		end
	end

	--Show/Hide symbols if symbol tags changed
	local tagschanged = false
	for tag in pairs(self.symboltags[bodypart]) do
		if oldsymboltags[bodypart][tag] then
			oldsymboltags[bodypart][tag] = nil
		else
			tagschanged = true
			break
		end
	end
	tagschanged = tagschanged or (oldsymboltags[bodypart] ~= nil and next(oldsymboltags[bodypart]) ~= nil)

	if tagschanged then
		for symbol, tag in pairs(Cosmetic.BodySymbolFilters) do
			if self.symboltags[bodypart][tag] then
				self.inst.AnimState:ShowSymbol(symbol)
			else
				self.inst.AnimState:HideSymbol(symbol)
			end
		end
	end

	--Check if filter tags changed
	tagschanged = false
	for tag in pairs(self.filtertags) do
		if oldfiltertags[tag] then
			oldfiltertags[tag] = nil
		else
			tagschanged = true
			break
		end
	end
	tagschanged = tagschanged or next(oldfiltertags) ~= nil

	self.inst:PushEvent("onbodypartchanged", { bodypart = bodypart, oldpart = oldpart, newpart = name })
	if self.use_playerdata_storage then
		ThePlayerData:SetCharacterCreatorBodyPart(self.inst.Network:GetPlayerID(), bodypart, name)
	end
	return tagschanged, true
end

function CharacterCreator:OverrideBodyPartSymbols(bodypart, build, colorgroup)
	if build == nil then
		self:ClearBodyPartSymbols(bodypart)
		return
	end

	local symbols = Cosmetic.BodySymbols[bodypart]
	if symbols ~= nil then
		for i = 1, #symbols do
			local symbol = symbols[i]
			self.inst.AnimState:OverrideSymbol(symbol, build, Cosmetic.BodySymbolRemaps[symbol] or symbol)

			local colorgroup1 = Cosmetic.SymbolColorOverrides[symbol] or colorgroup
			local colorname = self.colorgroups[colorgroup1]
			local color = colorname ~= nil and Cosmetic.Colors[colorgroup1][colorname] or nil
			if color ~= nil then
				self.inst.AnimState:SetSymbolColorShift(symbol, table.unpack(color.hsb))
			else
				self.inst.AnimState:ClearSymbolColorShift(symbol)
			end
		end
	end
end

function CharacterCreator:ClearBodyPartSymbols(bodypart)
	local symbols = Cosmetic.BodySymbols[bodypart]
	if symbols ~= nil then
		for i = 1, #symbols do
			local symbol = symbols[i]
			self.inst.AnimState:ClearOverrideSymbol(symbol)
			self.inst.AnimState:ClearSymbolColorShift(symbol)
		end
	end
end

function CharacterCreator:ClearAllBodyPartSymbols()
	local bodyparts = Cosmetic.BodyPartGroups
	for bodypart in pairs(bodyparts) do
		self:ClearBodyPartSymbols(bodypart)
	end
end

function CharacterCreator:GetColorList(colorgroup, filtered)
	return Cosmetic.GetColorList(colorgroup, filtered and self.filtertags or nil)
end

function CharacterCreator:GetColor(colorgroup)
	return self.colorgroups[colorgroup]
end

function CharacterCreator:SetColorGroup(colorgroup, colorname)
	if self.colorgroups[colorgroup] == colorname then
		return false
	end

	local colors = Cosmetic.Colors[colorgroup]
	if colors == nil then
		TheLog.ch.CharacterCreator:printf("Invalid color group: %s", colorgroup)
		return false
	end

	local color
	if colorname then
		color = colors[colorname]
		if color == nil then
			TheLog.ch.CharacterCreator:printf("Invalid %s color id: %s", colorgroup, colorname)
			return false
		end
	end

	if color and colorgroup == "SKIN_TONE" then
		self.inst:PushEvent("update_skin_color", color.rgb)
		if PER_PLAYER_SILHOUETTE_COLOR then
			local r,g,b,a = table.unpack(color.rgb)
			self.inst.AnimState:SetSilhouetteColor(r,g,b,PLAYER_SILHOUETTE_ALPHA)
		end
	end

	local oldcolor = self.colorgroups[colorgroup]
	self.colorgroups[colorgroup] = colorname
	if color then
		self:SetSymbolColorShift(colorgroup, table.unpack(color.hsb))
	end

	self.inst:PushEvent("oncolorchanged", { colorgroup = colorgroup, oldcolor = oldcolor, newcolor = colorname })
	if self.use_playerdata_storage then
		ThePlayerData:SetCharacterCreatorColorGroup(self.inst.Network:GetPlayerID(), colorgroup, colorname)
	end
	return true
end

function CharacterCreator:SetSymbolColorShift(colorgroup, hue, saturation, brightness)
	for bodypart, name in pairs(self.bodyparts) do
		if Cosmetic.BodyParts[bodypart][name].colorgroup == colorgroup then
			local symbols = Cosmetic.BodySymbols[bodypart]
			for i = 1, #symbols do
				local symbol = symbols[i]
				if Cosmetic.SymbolColorOverrides[symbol] == nil then
					self.inst.AnimState:SetSymbolColorShift(symbol, hue, saturation, brightness)
				end
			end
		end
	end
	for symbol, colorgroup1 in pairs(Cosmetic.SymbolColorOverrides) do
		if colorgroup == colorgroup1 then
			self.inst.AnimState:SetSymbolColorShift(symbol, hue, saturation, brightness)
		end
	end
end

function CharacterCreator:ClearSymbolColorShift(colorgroup)
	for bodypart, name in pairs(self.bodyparts) do
		if Cosmetic.BodyParts[bodypart][name].colorgroup == colorgroup then
			local symbols = Cosmetic.BodySymbols[bodypart]
			for i = 1, #symbols do
				local symbol = symbols[i]
				if Cosmetic.SymbolColorOverrides[symbol] == nil then
					self.inst.AnimState:ClearSymbolColorShift(symbols[i])
				end
			end
		end
	end
	for symbol, colorgroup1 in pairs(Cosmetic.SymbolColorOverrides) do
		if colorgroup == colorgroup1 then
			self.inst.AnimState:ClearSymbolColorShift(symbol)
		end
	end
end

function CharacterCreator:Randomize(species, owner, unlock_all)
	species = species or krandom.PickValue(CHARACTER_SPECIES)
	self:SetSpecies(species)

	for bodypart in pairs(Cosmetic.BodyPartGroups) do
		local bodyparts = Cosmetic.GetSpeciesBodyParts(bodypart, species)
		if not unlock_all then
			bodyparts = lume.removeall(bodyparts, function(bp)
				local unlocked = self:IsBodyPartUnlocked(bp, owner)
				return not unlocked
			end)
		end

		bodyparts = krandom.Shuffle(bodyparts)

		if #bodyparts > 0 then
			self:SetBodyPart(bodypart, bodyparts[1].name)
			local colorgroup = bodyparts[1].colorgroup
			if colorgroup ~= nil then
				local colors = Cosmetic.GetSpeciesColors(colorgroup, species)
				if not unlock_all then
					colors = lume.removeall(colors, function(clr)
						local unlocked = self:IsColorUnlocked(clr, owner)
						return not unlocked
					end)
				end

				colors = krandom.Shuffle(colors)
				assert(#colors > 0, "BAD COSMETIC SETUP, MISSING COLORS FOR GROUP:" .. colorgroup)
				self:SetColorGroup(colorgroup, colors[1].name)
			end
		else
			if bodypart ~= "OTHER" or (bodypart == "OTHER" and species == "canine") then
				print ("NO BODY PART FOUND FOR ", species, bodypart, "SETTING DEFAULT INSTEAD")
				for i, data in ipairs(DEFAULT_CHARACTERS_SETUP) do
					if data.species == species then
						local bodypart_name = data.bodyparts[bodypart]
						if bodypart_name ~= nil then
							self:SetBodyPart(bodypart, bodypart_name)
							local colorgroup = Cosmetic.BodyParts[bodypart][bodypart_name].colorgroup
							self:SetColorGroup(colorgroup, data.colorgroups[colorgroup])
						end
					end
				end
			end
		end
	end
end

function CharacterCreator:SaveToTable()
	local data = {}
	if next(self.bodyparts) ~= nil then
		data.bodyparts = {}
		for bodypart, name in pairs(self.bodyparts) do
			data.bodyparts[bodypart] = name
		end
	end
	if next(self.colorgroups) ~= nil then
		data.colorgroups = {}
		for colorgroup, id in pairs(self.colorgroups) do
			data.colorgroups[colorgroup] = id
		end
	end

	data.species = self.species

	return next(data) ~= nil and data or nil
end

-- currently functionally the same as LoadFromTable but some added checks
-- would treat other component as 'readonly'
function CharacterCreator:CloneComponent(other)
	assert(other:is_a(CharacterCreator))
	self:LoadFromTable(other)
end

function CharacterCreator:LoadFromTable(data)
	self.isnew = nil

	if data then
		local did_change

		self.species = data.species

		-- special case when importing legacy data: set this so it can propagate correctly
		if self.use_playerdata_storage then
			local playerID = self.inst.Network:GetPlayerID()
			if TheNet:IsValidPlayer(playerID) then
				ThePlayerData:SetCharacterCreatorSpecies(playerID, data.species)
			end
		end

		if data.bodyparts ~= nil then
			for bodypart, name in pairs(data.bodyparts) do
				local tags_changed, parts_changed = self:SetBodyPart(bodypart, name)
				did_change = did_change or parts_changed
			end
		end

		if data.colorgroups ~= nil then
			for colorgroup, name in pairs(data.colorgroups) do
				local colour_changed = self:SetColorGroup(colorgroup, name)
				did_change = did_change or colour_changed
			end
		end

		local colors_were_invalid = self:ValidateColorGroups()
		did_change = did_change or colors_were_invalid

		if did_change then
			self.inst:PushEvent("charactercreator_load")
		end
	end
end

function CharacterCreator:OnSave()
	if self.use_playerdata_storage and self.inst:IsLocal() then
		local playerID = self.inst.Network:GetPlayerID()
		if TheNet:IsValidPlayer(playerID) then
			if next(self.bodyparts) ~= nil then
				for bodypart, name in pairs(self.bodyparts) do
					ThePlayerData:SetCharacterCreatorBodyPart(playerID, bodypart, name)
				end
			end
			if next(self.colorgroups) ~= nil then
				for colorgroup, id in pairs(self.colorgroups) do
					ThePlayerData:SetCharacterCreatorColorGroup(playerID, colorgroup, id)
				end
			end

			ThePlayerData:SetCharacterCreatorSpecies(playerID, self.species)
		end
	end
end

function CharacterCreator:OnLoad(data)
	if data then
		self:LoadFromTable(data)

		if self.use_playerdata_storage then
			local msg = string.format("Character Creator has loaded legacy data for %s. Save game to store in new location.", self.inst:GetCustomUserName())
			TheLog.ch.CharacterCreator:print("******************************************************************")
			TheLog.ch.CharacterCreator:print("SaveSystem / OnLoad: " .. msg)
			TheLog.ch.CharacterCreator:print("******************************************************************")
			TheFrontEnd:ShowTextNotification("images/ui_ftf/warning.tex", nil, msg, 12)
		end
	end
end

function CharacterCreator:_OnPlayerDataChanged(data_changed_flags)
	if self.use_playerdata_storage and (data_changed_flags & PlayerDataChangedFlags.CharacterCreator) == PlayerDataChangedFlags.CharacterCreator then
		TheLog.ch.CharacterCreatorSpam:printf("%s OnPlayerDataChanged", self.inst)
		self.isnew = nil
		local playerID = self.inst.Network:GetPlayerID()
		local did_change

		self.species = ThePlayerData:GetCharacterCreatorSpecies(playerID)

		local bodyparts = ThePlayerData:GetCharacterCreatorBodyParts(playerID)
		if bodyparts then
			for bodypart, name in pairs(bodyparts) do
				local tags_changed, parts_changed = self:SetBodyPart(bodypart, name)
				did_change = did_change or parts_changed
			end
		end

		local colorgroups = ThePlayerData:GetCharacterCreatorColorGroups(playerID)
		if colorgroups then
			for colorgroup, name in pairs(colorgroups) do
				local colour_changed = self:SetColorGroup(colorgroup, name)
				did_change = did_change or colour_changed
			end
		end

		local colors_were_invalid = self:ValidateColorGroups()
		did_change = did_change or colors_were_invalid

		if did_change then
			self.inst:PushEvent("charactercreator_load")
		end
	end
end

function CharacterCreator:_RefreshPlayerData()
	if self.use_playerdata_storage then
		local playerID = self.inst.Network:GetPlayerID()
		-- dbassert(TheNet:IsValidPlayer(playerID))
		-- dbassert(ThePlayerData:IsCharacterCreatorDataValid(playerID))

		if TheNet:IsValidPlayer(playerID) and ThePlayerData:IsCharacterCreatorDataValid(playerID) then
			-- force refresh ... possible performance issue
			TheLog.ch.CharacterCreator:printf("Refreshing player data for player %d, %s", playerID, self.inst)
			self:_OnPlayerDataChanged(PlayerDataChangedFlags.CharacterCreator)
		end
	end
end

function CharacterCreator:GetDebugString()
	local str = ""
	-- local delim = "\nFilter Tags - "
	-- for tag in pairs(self.filtertags) do
	-- 	str = str..string.format("%s%s", delim, tag)
	-- 	delim = ", "
	-- end
	-- delim = "\nSymbol Tags - "
	-- for tag in pairs(self.symboltags) do
	-- 	str = str..string.format("%s%s", delim, tag)
	-- 	delim = ", "
	-- end
	-- for bodypart in pairs(Cosmetic.BodyPartGroups) do
	-- 	str = str.."\n\t["..bodypart.."]"
	-- 	local name = self.bodyparts[bodypart]
	-- 	if name ~= nil then
	-- 		str = str.." - "..name
	-- 		local def = Cosmetic.BodyParts[bodypart][name]
	-- 		if def.colorgroup ~= nil then
	-- 			str = str.." - #"..HexToStr(self.colorgroups[def.colorgroup])
	-- 		end
	-- 	end
	-- end
	return str
end

return CharacterCreator
